unit StyleView1;

{
  Demonstrate how to apply XSLT stylesheets to an XML document.
  Uses DOM interfaces from Microsoft XML parser.
  Requires MSXML v3 package from Microsoft.

  Copyright  Keith Wood (kbwood@iprimus.com.au)
  Written May 18, 2000.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ComCtrls, ExtCtrls, OleCtrls, SHDocVw, ActiveX, MSXML2_TLB;

type
  TfrmStylesheets = class(TForm)
    pnlControls: TPanel;
      Label1: TLabel;
      edtXML: TEdit;
      btnXML: TButton;
      Label3: TLabel;
      edtElement: TEdit;
      Label2: TLabel;
      edtXSLT: TEdit;
      btnXSLT: TButton;
      btnTransform: TButton;
      btnSave: TButton;
    pgcStylesheets: TPageControl;
      tabXML: TTabSheet;
        memXML: TRichEdit;
      tabDOM: TTabSheet;
        trvDOM: TTreeView;
      tabXSLT: TTabSheet;
        memXSLT: TRichEdit;
      tabHTML: TTabSheet;
        brsOutput: TWebBrowser;
      tabRTF: TTabSheet;
        memOutput: TRichEdit;
    dlgOpen: TOpenDialog;
    dlgSave: TSaveDialog;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure btnXMLClick(Sender: TObject);
    procedure btnXSLTClick(Sender: TObject);
    procedure edtXMLChange(Sender: TObject);
    procedure trvDOMChange(Sender: TObject; Node: TTreeNode);
    procedure btnTransformClick(Sender: TObject);
    procedure btnSaveClick(Sender: TObject);
  private
    XMLDoc: IXMLDOMDocument;
    XSLTDoc: IXMLDOMDocument;
    XMLNode: IXMLDOMNode;
    HTMLOutput: Boolean;
    procedure ClearTreeView;
  public
  end;

var
  frmStylesheets: TfrmStylesheets;

implementation

{$R *.DFM}

resourcestring
  NoDOM      = 'Couldn''t create the DOM';
  NoOutput   = 'Nothing generated by the transformation';
  Results    = 'Results';
  XMLFilter  = 'XML files (*.xml)|*.xml|All files (*.*)|*.*';
  XMLOpen    = 'Open XML data file';
  XSLTFilter = 'XSLT files (*.xsl)|*.xsl|All files (*.*)|*.*';
  XSLTOpen   = 'Open XSLT stylesheet file';
  HTMLFilter = 'HTML files (*.html,*.htm)|*.html;*.htm|All files (*.*)|*.*';
  HTMLSave   = 'Save HTML file';
  TextFilter = 'Text files (*.txt)|*.txt|Rich-text files (*.rtf)|*.rtf|' +
               'Comma-separated files (*.csv)|*.csv|All files (*.*)|*.*';
  TextSave   = 'Save text file';

{ TXMLNode --------------------------------------------------------------------}

type
  { Object wrapper for a node reference }
  TXMLNode = class(TObject)
  public
    Node: IXMLDOMNode;
    constructor Create(Node: IXMLDOMNode);
  end;

constructor TXMLNode.Create(Node: IXMLDOMNode);
begin
  inherited Create;
  Self.Node := Node;
end;

{ TfrmStylesheets -------------------------------------------------------------}

{ Format a DOM node for display }
function NodeDisplay(Node: IXMLDOMNode): string;
begin
  Result := Node.Text;
  Result := '<' + Node.NodeName + '>' + Copy(Result, 1, 30);
end;

{ Instantiate the DOMs }
procedure TfrmStylesheets.FormCreate(Sender: TObject);
begin
  XMLDoc             := CoDOMDocument.Create;
  XSLTDoc            := CoDOMDocument.Create;
  dlgOpen.InitialDir := ExtractFilePath(Application.ExeName);
  brsOutput.Navigate('about:blank');
end;

{ Release the DOMs }
procedure TfrmStylesheets.FormDestroy(Sender: TObject);
begin
  ClearTreeView;
  XMLNode := nil;
  XMLDoc  := nil;
  XSLTDoc := nil;
end;

{ Free up the objects within the tree view }
procedure TfrmStylesheets.ClearTreeView;
var
  Index: Integer;
begin
  for Index := 0 to trvDOM.Items.Count - 1 do
    TXMLNode(trvDOM.Items[Index].Data).Free;
  trvDOM.Items.Clear;
end;

{ Find an XML source file }
procedure TfrmStylesheets.btnXMLClick(Sender: TObject);

  { Load the DOM elements into a tree view recursively }
  procedure LoadElements(Node: IXMLDOMNode; Parent: TTreeNode);
  var
    Index: Integer;
    Current: TTreeNode;
  begin
    if (Node.nodeType = NODE_ELEMENT) or (Node.nodeType = NODE_DOCUMENT) then
    begin
      Current := trvDOM.Items.AddChildObject(Parent,
        NodeDisplay(Node), TXMLNode.Create(Node));
      for Index := 0 to Node.childNodes.length - 1 do
        LoadElements(Node.childNodes[Index], Current);
    end;
  end;

begin
  with dlgOpen do
  begin
    Filename := edtXML.Text;
    Filter   := XMLFilter;
    Title    := XMLOpen;
    if Execute then
    begin
      edtXML.Text := Filename;
      memXML.Lines.Clear;
      trvDOM.Items.BeginUpdate;
      try
        ClearTreeView;
        { Load the XML data }
        memXML.Lines.LoadFromFile(edtXML.Text);
        if not XMLDoc.load(edtXML.Text) then
          with XMLDoc.parseError do
          begin
            MessageDlg(Reason + ' at ' + IntToStr(Line) + ',' +
              IntToStr(LinePos), mtError, [mbOK], 0);
            Exit;
          end;
        { Load the DOM tree view }
        LoadElements(XMLDoc, nil);
        trvDOM.Items[0].Expand(True);
        trvDOM.TopItem := trvDOM.Items[0];
        trvDOMChange(trvDOM, trvDOM.Items[0]);
      finally
        trvDOM.Items.EndUpdate;
        pgcStylesheets.ActivePage := tabDOM;
      end;
    end;
  end;
end;

{ Find an XSLT stylesheet }
procedure TfrmStylesheets.btnXSLTClick(Sender: TObject);
begin
  with dlgOpen do
  begin
    Filename := edtXSLT.Text;
    Filter   := XSLTFilter;
    Title    := XSLTOpen;
    if Execute then
    begin
      edtXSLT.Text := Filename;
      memXSLT.Lines.Clear;
      { Load the XSLT stylesheet }
      memXSLT.Lines.LoadFromFile(edtXSLT.Text);
      HTMLOutput := (Pos('<html>', memXSLT.Lines.Text) > 0) or
        (Pos('<HTML>', memXSLT.Lines.Text) > 0);
      if not XSLTDoc.load(edtXSLT.Text) then
        with XSLTDoc.parseError do
        begin
          MessageDlg(Reason + ' at ' + IntToStr(Line) + ',' +
            IntToStr(LinePos), mtError, [mbOK], 0);
          Exit;
        end;
      pgcStylesheets.ActivePage := tabXSLT;
    end;
  end;
end;

{ Dis/enable merge button }
procedure TfrmStylesheets.edtXMLChange(Sender: TObject);
begin
  btnTransform.Enabled := ((edtXML.Text <> '') and (edtXSLT.Text <> ''));
end;

{ Select the node to operate on }
procedure TfrmStylesheets.trvDOMChange(Sender: TObject; Node: TTreeNode);
begin
  XMLNode         := TXMLNode(Node.Data).Node;
  edtElement.Text := NodeDisplay(XMLNode);
end;

{ Apply the stylesheet to the data and see the results }
procedure TfrmStylesheets.btnTransformClick(Sender: TObject);
var
  Output: WideString;
  TempStream: TStringStream;
  StreamAdapter: TStreamAdapter;
begin
  { Combine the two and display the results }
  Output := XMLNode.transformNode(XSLTDoc);
  if Output = '' then
    MessageDlg(NoOutput, mtError, [mbOK], 0);

  with memOutput.DefAttributes do
  begin
    { Reset default style }
    Color := memOutput.Font.Color;
    Name  := memOutput.Font.Name;
    Size  := memOutput.Font.Size;
    Style := memOutput.Font.Style;
  end;
  memOutput.PlainText := HTMLOutput;
  tabHTML.TabVisible  := HTMLOutput;
  tabRTF.TabVisible   := not HTMLOutput;
  TempStream          := TStringStream.Create(Output);
  try
    { Load into memo }
    TempStream.Position := 0;
    memOutput.Lines.LoadFromStream(TempStream);

    if HTMLOutput then
    begin
      { Load into browser }
      TempStream.Position := 0;
      StreamAdapter       := TStreamAdapter.Create(TempStream);
      (brsOutput.Document as IPersistStreamInit).Load(StreamAdapter);
      pgcStylesheets.ActivePage := tabHTML;
    end
    else
      { Show rich text memo }
      pgcStylesheets.ActivePage := tabRTF;
  finally
    TempStream.Free;
  end;
  btnSave.Enabled := True;
end;

{ Save the resulting output }
procedure TfrmStylesheets.btnSaveClick(Sender: TObject);
begin
  with dlgSave do
  begin
    if HTMLOutput then
    begin
      Filter := HTMLFilter;
      Title  := HTMLSave;
    end
    else
    begin
      Filter := TextFilter;
      Title  := TextSave;
    end;
    if Execute then
      memOutput.Lines.SaveToFile(Filename);
  end;
end;

end.
